#include "output.h"
#include "idda_main.h"

output::output(configdata* in_config, resultdata* in_result, demodata* in_data)
{
	config = in_config;
	result = in_result;
	data = in_data;
	out = new outputfile (config);
}

void output::open(char *filename)
{
	out->open (filename);
}

void output::close()
{
	out->close ();
}

output::~output()
{
	out->close ();
	delete[] out;
}

void output::general_info_html()
{
	if (config->no_times == true) { // ???
		return;
	}
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>General info</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
	if (config->no_times == false) {
		out->write ("<tr>\r\n");
		out->write ("<td><strong>Total playtime</strong></td>\r\n");
		out->write ("<td>%s</td>\r\n", timestr (result->get_total_time() / 10));
		out->write ("</tr>\r\n");
	}
	out->write ("<tr>\r\n");
	out->write ("<td><strong>Demos</strong></td>\r\n");
	out->write ("<td>%i</td>\r\n", result->get_demo_count());
	out->write ("</tr>\r\n");
	out->write ("<tr>\r\n");
	out->write ("<td><strong>Players</strong></td>\r\n");
	out->write ("<td>%i</td>\r\n", result->get_player_count());
	out->write ("</tr>\r\n");
	demodata_t* pos = data->get_first_of_type(102);
	out->write ("<tr>\r\n");
	out->write ("<td><strong>Gametype</strong></td>\r\n");
	if (pos->message[0] == 0) {
		out->write ("<td>standard/unknown</td>\r\n");
	} else {
		out->write ("<td>%s</td>\r\n", pos->message);
	}
	out->write ("</tr>\r\n");
	table_footer();
}

void output::clan_info()
{
	bool clans_exist = false;
	for (int i = 0; i < config->get_clan_count(); i++) {
		if ((result->get_clan_frags(i) > 0) || (result->get_clan_kills(i) > 0)) {
			clans_exist = true;
		}
	}
	if (clans_exist == false) {
		return;
	}
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>Clan info</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
	out->write ("<tr>\r\n");
	out->write ("<td><strong>Clan</strong></td>\r\n");
	out->write ("<td><strong>Frags</strong></td>\r\n");
	out->write ("<td><strong>Kills</strong></td>\r\n");
	out->write ("<td><strong>Suicides</strong></td>\r\n");
	out->write ("<td><strong>Deaths</strong></td>\r\n");
	out->write ("<td><strong>Teamkills</strong></td>\r\n");
	out->write ("</tr>\r\n");
	for (i = -1; i < config->get_clan_count(); i++) {
		if (result->get_clan_frags(i) > 0) {
			out->write ("<tr>\r\n");
			if (i > -1) {
				out->write ("<td>%s</td>\r\n", config->get_clan(i)->name);
			} else {
				out->write ("<td>(Unassigned)</td>\r\n");
			}
			out->write ("<td>%i</td>\r\n", result->get_clan_frags(i));
			out->write ("<td>%i</td>\r\n", result->get_clan_kills(i));
			out->write ("<td>%i</td>\r\n", result->get_clan_suicides(i));
			out->write ("<td>%i</td>\r\n", result->get_clan_deaths(i));
			if (i > -1) {
				out->write ("<td>%i</td>\r\n", result->get_clan_memberkills(i));
			} else {
				out->write ("<td>-</td>\r\n");
			}
			out->write ("</tr>\r\n");
		}
	}
    table_footer();
}

void output::player_info_html ()
{
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>Player info</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
    out->write ("<tr>\r\n");
    out->write ("<td><strong>Rank</strong></td>\r\n");
    out->write ("<td><strong>Name</strong></td>\r\n");
	if (config->no_times == false) {
		out->write ("<td><strong>Average life (h)</strong></td>\r\n");
		out->write ("<td><strong>Longest life (h)</strong></td>\r\n");
		out->write ("<td><strong>MKIOL<sup>*</sup></strong></td>\r\n");
		if (result->get_quad_total_kills() > 0) {
			out->write ("<td><strong>Quad time (h)</strong></td>\r\n");
			out->write ("<td><strong>Quad kills</strong></td>\r\n");
		}
	}
    out->write ("<td><strong>Easiest prey</strong></td>\r\n");
    out->write ("<td><strong>Worst enemy</strong></td>\r\n");
    out->write ("</tr>\r\n");
    for (int i = 1; i <= result->get_player_count(); i++) {
		out->write ("<tr valign=\"top\">\r\n<td>%i</td>\r\n", i);
		if (result->get_player(i)->name[0] != 0) {
			out->write ("<td>%s</td>\r\n", result->get_player(i)->name);
		} else {
			out->write ("<td>&nbsp;</td>\r\n");
		}
		//		out->write ("<td>%i/%i</td>\r\n", result->get_player_demo_count(i), result->demo_count());
		if (config->no_times == false) {
			player_ranking_value (so_lifetime_average, i);
			player_ranking_value (so_lifetime_longest, i);
			player_ranking_value (so_mkiol, i);
			if (result->get_quad_total_kills() > 0) {
				player_ranking_value (so_quadtime, i);
				player_ranking_value (so_quadkills, i);
			}
		}
		out->write ("<td valign=\"top\">");
		int preytemp = result->get_player_easiest_prey(i);
		if (preytemp > -1) {
			int prey = result->get_player_kills(i, preytemp, 0);
			int count = 0;
			for (int j = 1; j <= result->get_player_count(); j++) {
				if (result->get_player_kills(i,j,0) == prey) {
					if (count > 0) {
						out->write ("<br>");
					}
					out->write ("%s (<font color=\"#FF0000\">%i</font>)", 
						result->get_player(j)->name, result->get_player_kills(i,j,0));
					count++;
				}
			}
		} else {
			out->write ("-");
		}
		out->write ("</td>\r\n");
		out->write ("<td valign=\"top\">");
		int enemytemp = result->get_player_worst_enemy(i);
		if (enemytemp > -1) {
			int enemy = result->get_player_kills(enemytemp, i, 0);
			int count = 0;
			for (int j = 1; j <= result->get_player_count(); j++) {
				if (result->get_player_kills(j,i,0) == enemy) {
					if (count > 0) {
						out->write ("<br>");
					}
					out->write ("%s (<font color=\"#FF0000\">%i</font>)", 
						result->get_player(j)->name, result->get_player_kills(j,i,0));
					count++;
				}
			}
		} else {
			out->write ("-");
		}
		out->write ("</td>\r\n");
		out->write ("</tr>\r\n");
    }
    table_footer();
	if (config->no_times == false) {
		line_header();
		out->write ("<font size=\"2\">* MKIOL means \"Most Kills In One Life\"</font>");
		line_footer();
	}
}


void output::player_ranking_html () 
{
	line_header();
	out->write ("<strong>Player ranking</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
    out->write ("<tr>\r\n<td><strong>Rank</strong></td>\r\n");
	player_ranking_header (so_name);
	if ((config->is_lmctf == true) || (config->is_battle == true)) {
		out->write ("<td><strong>Team</strong></td>\r\n");
	}
	if (config->no_times == false) {
		player_ranking_header (so_ratio);
	}
	player_ranking_header (so_skill);
	player_ranking_header (so_frags);
	player_ranking_header (so_kills);
	player_ranking_header (so_waskilled);
	player_ranking_header (so_suicides);
	if (result->get_total_deaths(0) > 0) {
		player_ranking_header (so_deaths);
		player_ranking_header (so_total_deaths);
	}
	if (config->is_battle == true) {
		player_ranking_header (so_memberkills);
	}
	if (config->no_times == false) {
		player_ranking_header (so_fph);
		player_ranking_header (so_kph);
		player_ranking_header (so_dph);
		out->write ("<td><strong>Lifetime (h)</strong></td>\r\n");
//		out->write ("<td><strong>Lifetime (%%)</strong></td>\r\n");
	}
	out->write ("</tr>");
    for (int i = 1; i <= result->get_player_count(); i++) {
		out->write ("<tr>\r\n<td>%i</td>\r\n", i);
		player_ranking_value (so_name, i);
		if (config->is_lmctf == true) {
			switch (result->get_player(i)->team) {
			case 0:
				out->write ("<td>Blue</td>\r\n");
				break;
			case 1:
				out->write ("<td>Red</td>\r\n");
				break;
			default:
				out->write ("<td>(Unassigned)</td>\r\n");
				break;
			}
		}
		if (config->is_battle == true) {
			switch (result->get_player(i)->team) {
			case 0:
				out->write ("<td>A</td>\r\n");
				break;
			case 1:
				out->write ("<td>B</td>\r\n");
				break;
			default:
				out->write ("<td>(Unassigned)</td>\r\n");
				break;
			}
		}
		if (config->no_times == false) {
			player_ranking_value (so_ratio, i);
		}
		player_ranking_value (so_skill, i);
		player_ranking_value (so_frags, i);
		player_ranking_value (so_kills, i);
		player_ranking_value (so_waskilled, i);
		player_ranking_value (so_suicides, i);
		if (result->get_total_deaths(0) > 0) {
			player_ranking_value (so_deaths, i);
			player_ranking_value (so_total_deaths, i);
		}
		if (config->is_battle == true) {
			player_ranking_value (so_memberkills, i);
		}
		if (config->no_times == false) {
			player_ranking_value (so_fph, i);
			player_ranking_value (so_kph, i);
			player_ranking_value (so_dph, i);
			out->write ("<td>%s (%.0f %%)</td>\r\n", timestr (result->get_player(i)->lifetime_total / 10),
			100.0 * ((float) result->get_player(i)->lifetime_total / (float) result->get_total_time()));
		}
		out->write ("</tr>\r\n");
    }
    table_footer();
	if (config->no_times == false) {
		line_header();
		out->write ("<font size=\"2\">* Ratio is calculated as (KPH - DPH)</font><br>\r\n");
		out->write ("<font size=\"2\">* Skill is calculated as (kills / (total deaths + 1))</font>");
		line_footer();
	}
}

void output::kills_by_weapons_html (int weapon) 
{
	if (weapon == 0) {
		kills_by_weapons_html (1);
		kills_by_weapons_html (2);
		return;
	}
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>Player kills by weapons ");
	switch (weapon) {
	case 1: 
		out->write ("(absolute)");
		break;
	case 2:
		out->write ("(relative)");
		break;
	case 3:
		out->write ("(absolute and relative)");
		break;
	default:
		break;
	}
	out->write ("</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
    out->write ("<tr>\r\n");
    out->write ("<td><strong>Rank</strong></td>\r\n");
	out->write ("<td><strong>Player</strong></td>\r\n");
	//	out->write ("<td><strong>Usage</strong></td>\r\n");
	for (int i = 1; i < config->get_weapon_count(); i++) {
		if (result->get_total_kills (i) > 0) {
			switch (weapon) {
			case 1:
			case 2:
			case 3:
				out->write ("<td><strong>%s</strong></td>\r\n", config->get_weapon(i)->name);
				break;
			default:
				break;
			}
		}
	}
	if (weapon != 2) {
		out->write ("<td><strong>TOTAL</strong></td>\r\n");
	}
    out->write ("</tr>\r\n");
    for (i = 1; i <= result->get_player_count(); i++) {
		out->write ("<tr>\r\n");
		out->write ("<td>%i</td>\r\n", i);
		if (result->get_player(i)->name[0] != 0) {
			out->write ("<td>%s</td>\r\n", result->get_player(i)->name);
		} else {
			out->write ("<td>&nbsp;</td>\r\n");
		}
		for (int j = 1; j < config->get_weapon_count(); j++) {
			if (result->get_total_kills(j) > 0) {
				if (result->get_player_total_kills(i,j) == 0) {
					switch (weapon) {
					case 1:
					case 2:
					case 3:
						out->write ("<td>-</td>\r\n");
						break;
					default:
						break;
					}
				} else {
					out->write ("<td>");
					if (result->get_player_total_kills(i,result->get_player_favourite_weapon(i)) ==
						result->get_player_total_kills(i,j)) {
						out->write ("<font color=\"#FF0000\">");
					} 
					switch (weapon) {
					case 1:
						out->write ("%i", result->get_player_total_kills(i,j));
						break;
					case 2:
						out->write ("%.1f %%", 
							(float) 100.0 * result->get_player_total_kills(i,j) / result->get_player_total_kills(i,0));
						break;
					case 3:
						out->write ("%i (%.0f%%)", result->get_player_total_kills(i,j),
							(float) 100.0 * result->get_player_total_kills(i,j) / result->get_player_total_kills(i,0));
						break;
					default:
						break;
					}
					if (result->get_total_kills(result->get_player_favourite_weapon(i)) ==
						result->get_total_kills(j)) {
						out->write ("</font>");
					}
					out->write ("</td>\r\n");
				}
			}
		}
		if (weapon != 2) {
			out->write ("<td><strong>%i</strong></td>\r\n", result->get_player_total_kills(i,0));
		}
		out->write ("</tr>\r\n");
    }
	out->write ("<tr>\r\n<td><strong>TOTAL</strong></td>\r\n");
	out->write ("<td>&nbsp;</td>\r\n");
	for (i = 1; i < config->get_weapon_count(); i++) {
		if (result->get_total_kills(i) > 0) {
			out->write ("<td><strong>");
			if (result->get_total_kills(result->get_favourite_weapon()) ==
				result->get_total_kills(i)) {
				out->write ("<font color=\"#FF0000\">");
			} 
			switch (weapon) {
			case 1:
				out->write ("%i", result->get_total_kills(i));
				break;
			case 2:
				out->write ("%.1f %%", 
					(float) 100.0 * result->get_total_kills(i) / result->get_total_kills(0));
				break;
			case 3:
				out->write ("%i (%.0f%%)", result->get_total_kills(i),
					(float) 100.0 * result->get_total_kills(i) / result->get_total_kills(0));
				break;
			default:
				break;
			}
			if (result->get_total_kills(result->get_favourite_weapon()) ==
				result->get_total_kills(i)) {
				out->write ("</font>");
			} 
			out->write ("</strong></td>\r\n");
		}
	}
	if (weapon != 2) {
		out->write ("<td><strong>%i</strong></td>\r\n", result->get_total_kills(0));
	}
	out->write ("</tr>\r\n");
    table_footer();
}

void output::deaths_html () 
{
	if (result->get_total_deaths(0) <= 0) {
		return;
	}
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>Player deaths</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
	out->write ("<tr>\r\n");
	out->write ("<td><strong>Rank</strong></td>\r\n");
	out->write ("<td><strong>Player</strong></td>\r\n");
	for (int i = 0; i < config->get_deathtype_count(); i++) {
		if (result->get_total_deaths (i) > 0) {
			out->write ("<td><strong>%s</strong></td>\r\n", config->get_deathtype(i)->name);
		}
	}
	out->write ("</tr>\r\n");
	for (i = 1; i <= result->get_player_count(); i++) {
		out->write ("<tr>\r\n");
		out->write ("<td>%i</td>", i);
		if (result->get_player(i)->name[0] != 0) {
			out->write ("<td>%s</td>\r\n", result->get_player(i)->name);
		} else {
			out->write ("<td>&nbsp;</td>\r\n");
		}
		for (int j = 0; j < config->get_deathtype_count(); j++) {
			if (result->get_total_deaths (j) > 0) {
				out->write ("<td>");
				if (result->get_max_player_deaths(j) == result->get_player_deaths(i,j)) {
					out->write ("<font color=\"#FF0000\">");
				}
				if (j == 0) {
					out->write ("<strong>%i</strong>", result->get_player_deaths (i, j));
				} else {
					out->write ("%i", result->get_player_deaths (i, j));
				}
				if (result->get_max_player_deaths(j) == result->get_player_deaths(i,j)) {
					out->write ("</font>");
				}
				out->write ("</td>\r\n");
			}
		}
		out->write ("</tr>\r\n");
	}
	table_footer();
}

void output::kills_html (int weapon) 
{
	if (config->no_killcharts == true) {
		return;
	}
    int i;
    int j;
    int s = 0;
	line_header();
    out->write ("<strong>Killchart (");
    out->write ("%s)</strong>", config->get_weapon(weapon)->name);
	line_footer();
    table_header("border=\"1\" cellspacing=\"1\"");
    out->write ("<tr>\r\n<td>&nbsp;</td>\r\n<td>&nbsp;</td>\r\n");
    for (i = 1; i <= result->get_player_count(); i++) {
        out->write ("<td><p align=\"center\"><font color=\"#FF0000\"><strong>%i</strong></font></p></td>\r\n", i);
    }
    out->write ("<td><strong>KILLS</strong></td>\r\n");
    for (i = 1; i <= result->get_player_count(); i++) {
		if (result->get_player(i)->name[0] != 0) {
			out->write ("<tr>\r\n<td>%s</td>\r\n", result->get_player(i)->name);
		} else {
			out->write ("<tr>\r\n<td>&nbsp;</td>\r\n");
		}
        out->write ("<td width=\"20\"><p align=\"center\"><font color=\"#FF0000\"><strong>%i</strong></font></p></td>\r\n", i);
        for (j = 1; j <= result->get_player_count(); j++) {
			if (i != j) {
				out->write ("<td width=\"20\"><p align=\"right\">%i</p></td>\r\n", result->get_player_kills(i,j,weapon));
			} else {
				out->write ("<td width=\"20\"><p align=\"right\"><font color=\"#0000FF\">%i</font></p></td>\r\n", result->get_player_kills(i,j,weapon));
			}
        }
        out->write ("<td><p align=\"right\"><strong>%i</strong></p></td>\r\n", result->get_player_total_kills (i,weapon));
    }
    out->write ("<tr>\r\n<td><strong>DEATHS</strong></td>\r\n");
	out->write ("<td>&nbsp;</td>\r\n");
    for (i = 1; i <= result->get_player_count(); i++) {
        out->write ("<td><p align=\"right\"><strong>%i</strong></p></td>\r\n", result->get_player_total_waskilled (i, weapon));
    }
    out->write ("<td><p align=\"right\"><strong>%i</strong></p></td>\r\n", result->get_total_kills (weapon));
    out->write ("</tr>\r\n");
	table_footer();
	line_header(); 
	line_footer();
}

void output::empty_html()
{
	line_header();
	out->write ("<strong>No kills, no results! :)</strong>");
	line_footer();
}

char* output::timestr(int seconds)
{
	static char str[10];
	str[0] = 0;
	sprintf (str, "%02i:%02i:%02i", time_h(seconds), time_m(seconds), time_s(seconds));
	return str;
}

int output::time_h(int seconds)
{
	return (seconds / 3600);
}

int output::time_m(int seconds)
{
	while (seconds >= 3600) {
		seconds -= 3600;
	}
	return (seconds / 60);
}

int output::time_s(int seconds)
{
	while (seconds >= 3600) {
		seconds -= 3600;
	}
	while (seconds >= 60) {
		seconds -= 60;
	}
	return seconds;
}


void output::player_info_ctf()
{
	if (config->is_lmctf == false) {
		return;
	}
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>Player CTF info</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
    out->write ("<tr>\r\n");
    out->write ("<td><strong>Name</strong></td>\r\n");
    out->write ("<td><strong>Team</strong></td>\r\n");
    out->write ("<td><strong>Captures</strong></td>\r\n");
    out->write ("<td><strong>Assists</strong></td>\r\n");
    out->write ("<td><strong>Returns</strong></td>\r\n");
    out->write ("<td><strong>Defends</strong></td>\r\n");
    out->write ("<td><strong>EFCKills<sup>*</sup></strong></td>\r\n");
    out->write ("<td><strong>Steals</strong></td>\r\n");
    out->write ("<td><strong>Drops</strong></td>\r\n");
    out->write ("<td><strong>CTF score</strong></td>\r\n");
    out->write ("<td><strong>Frags</strong></td>\r\n");
    out->write ("<td><strong>Total score</strong></td>\r\n");
    out->write ("</tr>\r\n");
    for (int i = 1; i <= result->get_player_count(); i++) {
		out->write ("<tr>\r\n");
		if (result->get_player(i)->name[0] != 0) {
			out->write ("<td>%s</td>\r\n", result->get_player(i)->name);
		} else {
			out->write ("<td>&nbsp;</td>\r\n");
		}
		switch (result->get_player(i)->team) {
		case 0:
			out->write ("<td>Blue</td>\r\n");
			break;
		case 1:
			out->write ("<td>Red</td>\r\n");
			break;
		default:
			out->write ("<td>(Unassigned)</td>\r\n");
			break;
		}
		out->write ("<td>%i (<font color=\"#FF0000\">%i</font>)</td>\r\n", 
			result->get_player_flag_captures(i),
			result->get_player_flag_captures_score(i));
		out->write ("<td><font color=\"#FF0000\">%i</font></td>\r\n", 
			result->get_player_ctf_assists(i));
		out->write ("<td>%i (<font color=\"#FF0000\">%i</font>)</td>\r\n", 
			result->get_player_flag_returns(i),
			result->get_player_flag_returns(i) + result->get_player_ctf_helped_return(i));
		out->write ("<td>%i (<font color=\"#FF0000\">%i</font>)</td>\r\n", 
			result->get_player_ctf_defends(i),
			result->get_player_ctf_defends_score(i));
		out->write ("<td>%i (<font color=\"#FF0000\">%i</font>)</td>\r\n", 
			result->get_player_flagcarrier_frags(i),
			result->get_player_flagcarrier_frags(i) * 2);
		out->write ("<td>%i</td>\r\n", result->get_player_flag_steals(i));
		out->write ("<td>%i</td>\r\n", result->get_player_flag_drops(i));
		out->write ("<td><font color=\"#00FF00\">%i</font></td>\r\n", 
			result->get_player_ctf_score(i));
		out->write ("<td><font color=\"#00FF00\">%i</font></td>\r\n", 
			result->get_player_frags(i));
		out->write ("<td><font color=\"#0000FF\">%i</font></td>\r\n", 
			result->get_player_ctf_score(i) + result->get_player_frags(i));
		out->write ("</tr>\r\n");
    }
    table_footer();
	line_header();
	out->write ("<font size=\"2\">* EFCKills means \"Enemy Flag Carrier Kills\"</font>");
	line_footer();
}

void output::team_info_ctf()
{
	if (config->is_lmctf == false) {
		return;
	}
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>Team CTF info</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
    out->write ("<tr>\r\n");
    out->write ("<td><strong>Team</strong></td>\r\n");
    out->write ("<td><strong>Players</strong></td>\r\n");
    out->write ("<td><strong>Captures</strong></td>\r\n");
    out->write ("<td><strong>Assists</strong></td>\r\n");
    out->write ("<td><strong>Returns</strong></td>\r\n");
    out->write ("<td><strong>Defends</strong></td>\r\n");
    out->write ("<td><strong>EFCKills<sup>*</sup></strong></td>\r\n");
    out->write ("<td><strong>Steals</strong></td>\r\n");
    out->write ("<td><strong>Drops</strong></td>\r\n");
    out->write ("<td><strong>CTF score</strong></td>\r\n");
    out->write ("<td><strong>Frags</strong></td>\r\n");
    out->write ("<td><strong>Total score</strong></td>\r\n");
    out->write ("</tr>\r\n");
    for (int i = 0; i < 2; i++) {
		out->write ("<tr>\r\n");
		switch (i) {
		case 0:
			out->write ("<td>Blue</td>\r\n");
			break;
		case 1:
			out->write ("<td>Red</td>\r\n");
			break;
		}
		out->write ("<td>%i</td>\r\n", result->get_team_member_count(i));
		out->write ("<td>%i (<font color=\"#FF0000\">%i</font>)</td>\r\n", 
			result->get_team_flag_captures(i), 
			result->get_team_flag_captures(i) * (result->get_team_member_count(i) * 10 + 5));
		out->write ("<td><font color=\"#FF0000\">%i</font></td>\r\n", 
			result->get_team_ctf_assists(i));
		out->write ("<td>%i (<font color=\"#FF0000\">%i</font>)</td>\r\n", 
			result->get_team_flag_returns(i),
			result->get_team_flag_returns(i) + result->get_team_ctf_helped_return(i));
		out->write ("<td>%i (<font color=\"#FF0000\">%i</font>)</td>\r\n", 
			result->get_team_ctf_defends(i), result->get_team_ctf_defends_score(i));
		out->write ("<td>%i (<font color=\"#FF0000\">%i</font>)</td>\r\n", 
			result->get_team_flagcarrier_frags(i),
			result->get_team_flagcarrier_frags(i) * 2);
		out->write ("<td>%i</td>\r\n", result->get_team_flag_steals(i));
		out->write ("<td>%i</td>\r\n", result->get_team_flag_drops(i));
		out->write ("<td><font color=\"#00FF00\">%i</font></td>\r\n", 
			result->get_team_ctf_score(i));
		out->write ("<td><font color=\"#00FF00\">%i</font></td>\r\n", 
			result->get_team_frags(i));
		out->write ("<td><font color=\"#0000FF\">%i</font></td>\r\n", 
			result->get_team_ctf_score(i) + result->get_team_frags(i));
		out->write ("</tr>\r\n");
    }
    table_footer();
	line_header();
	out->write ("<font size=\"2\">* EFCKills means \"Enemy Flag Carrier Kills\"</font>");
	line_footer();
}

void output::team_info()
{
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>Team info</strong>");
	line_footer();
	table_header("border=\"1\" cellspacing=\"1\"");
    out->write ("<tr>\r\n");
    out->write ("<td><strong>Team</strong></td>\r\n");
    out->write ("<td><strong>Players</strong></td>\r\n");
    out->write ("<td><strong>Score</strong></td>\r\n");
    out->write ("<td><strong>Frags</strong></td>\r\n");
    out->write ("<td><strong>Kills</strong></td>\r\n");
    out->write ("<td><strong>Killed</strong></td>\r\n");
    out->write ("<td><strong>Suicides</strong></td>\r\n");
    out->write ("<td><strong>Deaths</strong></td>\r\n");
    out->write ("<td><strong>Total deaths</strong></td>\r\n");
    out->write ("<td><strong>Teamkills</strong></td>\r\n");
	if (result->get_quad_total_kills() > 0) {
		out->write ("<td><strong>Quadtime</strong></td>\r\n");
		out->write ("<td><strong>Quadkills</strong></td>\r\n");
	}
    out->write ("<td><strong>Skill<sup>*</sup></strong></td>\r\n");
    out->write ("</tr>\r\n");
    for (int i = 0; i < 2; i++) {
		out->write ("<tr>\r\n");
		switch (i) {
		case 0:
			out->write ("<td>A</td>\r\n");
			break;
		case 1:
			out->write ("<td>B</td>\r\n");
			break;
		}
		out->write ("<td>%i</td>\r\n", result->get_team_member_count(i));
		out->write ("<td><font color=\"#FF0000\"><strong>%i</strong></font></td>\r\n", 
			result->get_team_frags(i) - result->get_team_memberkills(i));
		out->write ("<td><font color=\"#00FF00\">%i</font></td>\r\n", 
			result->get_team_frags(i));
		out->write ("<td><font color=\"#FFFF00\">%i</font></td>\r\n", 
			result->get_team_kills(i));
		out->write ("<td>%i</td>\r\n", result->get_team_killed(i));
		out->write ("<td><font color=\"#FFFF00\">%i</font></td>\r\n", 
			result->get_team_suicides(i));
		out->write ("<td><font color=\"#FFFF00\">%i</font></td>\r\n", 
			result->get_team_deaths(i));
		out->write ("<td>%i</td>\r\n", result->get_team_total_deaths(i));
		out->write ("<td><font color=\"#00FF00\">%i</font></td>\r\n", 
			result->get_team_memberkills(i));
		if (result->get_quad_total_kills() > 0) {
			out->write ("<td>%s (%.0f %%)</td>", 
				timestr (result->get_team_quadtime(i) / 10), 
				100.0 * (double) result->get_team_quadtime(i)
				/ (double) result->get_total_time());
			out->write ("<td>%i (%.0f %%)</td>", 
				result->get_team_quadkills(i),
				100.0 * (double) result->get_team_quadkills(i) 
				/ (double) result->get_team_kills(i));
		}
		if (result->get_team_total_deaths(i) > 0) {
			out->write ("<td>%.2f</td>\r\n", (double) result->get_team_kills(i) 
				/ ((double) result->get_team_total_deaths(i) + 1));
		} else {
			out->write ("<td>0 %%</td>\r\n");
		}
		out->write ("</tr>\r\n");
    }
    table_footer();
	line_header();
	out->write ("<font size=\"2\">* Skill is calculated as (kills / (total deaths + 1))</font>\r\n");
	line_footer();
}

void output::sort_players(sort_t order1, sort_t order2)
{
	result->sort_players (order2);
	result->sort_players (order1);
	if (config->top_x_players > 0) {
		result->set_player_count (config->top_x_players);
	}
	line_header();
	out->write ("(Players sorted by ");
	sort_t orders[2];
	orders[0] = order1;
	orders[1] = order2;
	for (int i = 0; i < 2; i++) {
		if (i == 1) out->write (", ");
		switch (orders[i]) {
		case so_name:
			out->write ("name");
			break;
		case so_clan:
			out->write ("clan");
			break;
		case so_team:
			out->write ("team");
			break;
		case so_ctf_score:
			out->write ("score");
			break;
		case so_ratio:
			out->write ("ratio");
			break;
		case so_skill:
			out->write ("skill");
			break;
		case so_frags:
			out->write ("frags");
			break;
		case so_kills:
			out->write ("kills");
			break;
		case so_memberkills:
			out->write ("memberkills");
			break;
		case so_quadtime:
			out->write ("quadtime");
			break;
		case so_quadkills:
			out->write ("quadkills");
			break;
		case so_fph:
			out->write ("FPH");
			break;
		case so_kph:
			out->write ("KPH");
			break;
		default:
			out->write ("???");
			break;
		}
	}
	out->write (")");
	line_footer();
}

void output::player_ranking_header(sort_t order)
{
	out->write ("<td><strong>");
	switch (order) {
	case so_name:
		out->write ("Name");
		break;
	case so_ratio:
		out->write ("Ratio<sup>*</sup>");
		break;
	case so_skill:
		out->write ("Skill<sup>*</sup>");
		break;
	case so_frags:
		out->write ("Frags");
		break;
	case so_kills:
		out->write ("Kills");
		break;
	case so_memberkills:
		out->write ("Teamkills");
		break;
	case so_waskilled:
		out->write ("Killed");
		break;
	case so_suicides:
		out->write ("Suicides");
		break;
	case so_deaths:
		out->write ("Deaths");
		break;
	case so_total_deaths:
		out->write ("Total deaths");
		break;
	case so_fph:
		out->write ("FPH");
		break;
	case so_kph:
		out->write ("KPH");
		break;
	case so_dph:
		out->write ("DPH");
		break;
	}
	out->write ("</strong></td>\r\n");
}

void output::player_ranking_value(sort_t order, int i)
{
	int color_high[3] = { 0, 255, 0 };
	int color_middle[3] = { 255, 255, 0 };
	int color_low[3] = { 255, 0, 0 };
	out->write ("<td>");
	switch (order) {
	case so_name:
		if (result->get_player(i)->name[0] != 0) {
			out->write ("%s", result->get_player(i)->name);
		} else {
			out->write ("&nbsp;");
		}
		break;
	case so_ratio:
		out->write ("<font color=\"#%s\">%.2f</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_ratio), result->get_player_count()),
			result->get_player_ratio(i));
		break;
	case so_skill:
		out->write ("<font color=\"#%s\">%.2f</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_skill), result->get_player_count()),
			result->get_player_skill(i));
		break;
	case so_frags:
		out->write ("<font color=\"#%s\">%i</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_frags), result->get_player_count()),
			result->get_player_frags(i));
		break;
	case so_kills:
		out->write ("<font color=\"#%s\">%i</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_kills), result->get_player_count()),
			result->get_player_total_kills(i,0));
		break;
	case so_memberkills:
		out->write ("<font color=\"#%s\">%i</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_memberkills), result->get_player_count()),
			result->get_player_memberkills(i,0));
		break;
	case so_quadtime:
		out->write ("<font color=\"#%s\">%s (%.0f %%)</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_quadtime), result->get_player_count()),
			timestr (result->get_player(i)->quad_time / 10), 
			100.0 * (double) result->get_player(i)->quad_time / (double) result->get_player_lifetime_total(i));
		break;
	case so_quadkills:
		if (result->get_player(i)->quad_kills == 0) {
			out->write ("<font color=\"#%s\">%i (%.0f %%)</font>", 
				get_color_string (color_high, color_low, result->get_player_rank(i,so_quadkills), result->get_player_count()),
				0, 0);
		} else {
			out->write ("<font color=\"#%s\">%i (%.0f %%)</font>", 
				get_color_string (color_high, color_low, result->get_player_rank(i,so_quadkills), result->get_player_count()),
				result->get_player(i)->quad_kills,
				100.0 * (double) result->get_player(i)->quad_kills / (double) result->get_player_total_kills(i,0));
		}
		break;
	case so_waskilled:
		out->write ("<font color=\"#%s\">%i</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_waskilled), result->get_player_count()),
			result->get_player_total_waskilled(i,0));
		break;
	case so_suicides:
		out->write ("<font color=\"#%s\">%i</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_suicides), result->get_player_count()),
			result->get_player_suicides(i,0));
		break;
	case so_deaths:
		out->write ("<font color=\"#%s\">%i</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_deaths), result->get_player_count()),
			result->get_player_deaths(i,0));
		break;
	case so_total_deaths:
		out->write ("<font color=\"#%s\">%i</font>", 
			get_color_string (color_high, color_low, result->get_player_rank(i,so_total_deaths), result->get_player_count()),
			result->get_player_total_deaths(i));
		break;
	case so_fph:
		out->write ("<font color=\"#%s\">%.2f</font>", get_color_string (color_high, color_low, result->get_player_rank(i,so_fph), result->get_player_count()),
			result->get_player_fph(i));
		break;
	case so_kph:
		out->write ("<font color=\"#%s\">%.2f</font>", get_color_string (color_high, color_low, result->get_player_rank(i,so_kph), result->get_player_count()),
			result->get_player_kph(i));
		break;
	case so_dph:
		out->write ("<font color=\"#%s\">%.2f</font>", get_color_string (color_high, color_low, result->get_player_rank(i,so_dph), result->get_player_count()),
			result->get_player_dph(i));
		break;
	case so_lifetime_longest:
		out->write ("<font color=\"#%s\">%s</font>", get_color_string (color_high, color_low, result->get_player_rank(i,so_lifetime_longest), result->get_player_count()),
			timestr (result->get_player_lifetime_longest(i) / 10));
		break;
	case so_lifetime_average:
		out->write ("<font color=\"#%s\">%s</font>", get_color_string (color_high, color_low, result->get_player_rank(i,so_lifetime_average), result->get_player_count()),
			timestr (result->get_player_lifetime_average(i) / 10));
		break;
	case so_mkiol:
		out->write ("<font color=\"%s\">%i</font>", get_color_string (color_high, color_low, result->get_player_rank(i,so_mkiol), result->get_player_count()),
			result->get_player_mkiol(i));
		break;
	}
	out->write ("</td>\r\n");
}

void output::demos_html()
{
	if (config->no_times == true) {
		return;
	}
	line_header();
	line_footer();
	line_header();
	out->write ("<strong>Demo info</strong>");
	line_footer();
	for (int i = 0; i < result->get_demo_count(); i++) {
		if (config->centerprint == false) {
			demo_t* current_demo = result->get_demo (i);
			out->write ("<p>%s [%i byte, %s hours]</p>\n<table border=\"0\">\n", 
				current_demo->name, current_demo->size, timestr(current_demo->seconds));
			map_t* current_map = current_demo->maps->next;
			for (int j = 0; j < current_demo->mapcount; j++) {
				out->write ("<tr><td width=\"50\"></td><td width=\"300\">%s</td><td>[%s - %s]</td></tr>\n", 
					current_map->name, timestr(current_map->seconds), current_map->file);
				current_map = current_map->next;
			}
			if (current_demo->mapcount > 0) {
				out->write ("</table>\n");
			}
		} else {
			demo_t* current_demo = result->get_demo (i);
			line_header();
			table_header ("border=\"1\" cellspacing=\"1\" width=\"600\"");
			out->write ("<tr>\r\n");
			out->write ("<td colspan=\"3\" width=\"50%%\"><p align=\"center\">");
			out->write ("<strong>%s [%i byte, %s hours]</strong></p></td>\r\n", 
				current_demo->name, current_demo->size, timestr(current_demo->seconds));
			out->write ("</tr>\r\n");
			map_t* current_map = current_demo->maps->next;
			for (int j = 0; j < current_demo->mapcount; j++) {
				out->write ("<tr>\r\n");
				out->write ("<td width=\"33%%\">%s</td>", current_map->name);
				out->write ("<td width=\"33%%\"><p align=\"center\">%s</p></td>", 
					current_map->file);
				out->write ("<td width=\"34%%\"><p align=\"right\">%s h</p></td>",
					timestr(current_map->seconds));
				current_map = current_map->next;
			}
			table_footer();
		}
	}
}

char* output::get_color_string(int* color_high, int* color_low, int rank, int count)
{
	double gamma = 1 - ((double) rank - 1.0) * (double) 1.0 / ((double) count - 1.0);
	int color[3];
	for (int i = 0; i < 3; i++) {
		color[i] = color_low[i] + ((color_high[i] - color_low[i]) * gamma);
	}
	static char colorstr[7];
	sprintf (colorstr, "%02X%02X%02X", color[0], color[1], color[2]);
	return colorstr;
}


void output::kpt_players()
{
	write_graph graph;
	player_t* player;
	int pcount = result->get_player_count();
	if (pcount > 0) {
		line_header();
		for (int i = 1; i <= pcount; i++) {
			player = result->get_player(i);
			int* rgb = graph.get_rgb (i+1, pcount);
			out->write ("<font color=\"#%02X%02X%02X\">%s</font><br>\r\n",
				player->stats_color[0], player->stats_color[1], player->stats_color[2],
				player->name);
		}
		line_footer();
	}
}

void output::table_header(char* formatstring)
{
	if (config->centerprint == true) {
		out->write ("<div align=\"center\"><center>\r\n"); // centered
	}
	out->write ("<table %s>", formatstring);
}

void output::table_footer()
{
	out->write ("</table>");
	if (config->centerprint == true) {
		out->write ("</center></div>\r\n"); // centered
	}
}

void output::line_header()
{
	if (config->centerprint == false) {
		out->write ("<p>");
	} else {
		out->write ("<p align=\"center\">");
	}
}

void output::line_footer()
{
	out->write ("</p>");
}

void output::html_header()
{
	out->write ("<html>\r\n");
	out->write ("<body bgcolor=\"#000000\" text=\"#FFFFFF\">\r\n");
}

void output::html_footer()
{
	line_header();
	out->write ("<font size=\"2\"><a href=\"http://www.planetquake.com/idda/\">");
	out->write ("<font color=\"#FF0000\">idDA</font></a> %s, %s", version, date);
	out->write (" by <a href=\"mailto:spike@planetquake.com\">");
	out->write ("<font color=\"#FFFFFF\">Fabian Huester</font></font>");
	line_footer();
	out->write ("</body>\r\n");
	out->write ("</html>\r\n");
}
